<?php

declare (strict_types=1);
namespace PYS_PRO_GLOBAL\GuzzleHttp\Psr7;

use PYS_PRO_GLOBAL\Psr\Http\Message\StreamInterface;
/**
 * Stream that when read returns bytes for a streaming multipart or
 * multipart/form-data stream.
 */
final class MultipartStream implements \PYS_PRO_GLOBAL\Psr\Http\Message\StreamInterface
{
    use StreamDecoratorTrait;
    /** @var string */
    private $boundary;
    /**
     * @param array  $elements Array of associative arrays, each containing a
     *                         required "name" key mapping to the form field,
     *                         name, a required "contents" key mapping to a
     *                         StreamInterface/resource/string, an optional
     *                         "headers" associative array of custom headers,
     *                         and an optional "filename" key mapping to a
     *                         string to send as the filename in the part.
     * @param string $boundary You can optionally provide a specific boundary
     *
     * @throws \InvalidArgumentException
     */
    public function __construct(array $elements = [], string $boundary = null)
    {
        $this->boundary = $boundary ?: \sha1(\uniqid('', \true));
        $this->stream = $this->createStream($elements);
    }
    public function getBoundary() : string
    {
        return $this->boundary;
    }
    public function isWritable() : bool
    {
        return \false;
    }
    /**
     * Get the headers needed before transferring the content of a POST file
     *
     * @param array<string, string> $headers
     */
    private function getHeaders(array $headers) : string
    {
        $str = '';
        foreach ($headers as $key => $value) {
            $str .= "{$key}: {$value}\r\n";
        }
        return "--{$this->boundary}\r\n" . \trim($str) . "\r\n\r\n";
    }
    /**
     * Create the aggregate stream that will be used to upload the POST data
     */
    protected function createStream(array $elements = []) : \PYS_PRO_GLOBAL\Psr\Http\Message\StreamInterface
    {
        $stream = new \PYS_PRO_GLOBAL\GuzzleHttp\Psr7\AppendStream();
        foreach ($elements as $element) {
            $this->addElement($stream, $element);
        }
        // Add the trailing boundary with CRLF
        $stream->addStream(\PYS_PRO_GLOBAL\GuzzleHttp\Psr7\Utils::streamFor("--{$this->boundary}--\r\n"));
        return $stream;
    }
    private function addElement(\PYS_PRO_GLOBAL\GuzzleHttp\Psr7\AppendStream $stream, array $element) : void
    {
        foreach (['contents', 'name'] as $key) {
            if (!\array_key_exists($key, $element)) {
                throw new \InvalidArgumentException("A '{$key}' key is required");
            }
        }
        $element['contents'] = \PYS_PRO_GLOBAL\GuzzleHttp\Psr7\Utils::streamFor($element['contents']);
        if (empty($element['filename'])) {
            $uri = $element['contents']->getMetadata('uri');
            if (\substr($uri, 0, 6) !== 'php://') {
                $element['filename'] = $uri;
            }
        }
        [$body, $headers] = $this->createElement($element['name'], $element['contents'], $element['filename'] ?? null, $element['headers'] ?? []);
        $stream->addStream(\PYS_PRO_GLOBAL\GuzzleHttp\Psr7\Utils::streamFor($this->getHeaders($headers)));
        $stream->addStream($body);
        $stream->addStream(\PYS_PRO_GLOBAL\GuzzleHttp\Psr7\Utils::streamFor("\r\n"));
    }
    private function createElement(string $name, \PYS_PRO_GLOBAL\Psr\Http\Message\StreamInterface $stream, ?string $filename, array $headers) : array
    {
        // Set a default content-disposition header if one was no provided
        $disposition = $this->getHeader($headers, 'content-disposition');
        if (!$disposition) {
            $headers['Content-Disposition'] = $filename === '0' || $filename ? \sprintf('form-data; name="%s"; filename="%s"', $name, \basename($filename)) : "form-data; name=\"{$name}\"";
        }
        // Set a default content-length header if one was no provided
        $length = $this->getHeader($headers, 'content-length');
        if (!$length) {
            if ($length = $stream->getSize()) {
                $headers['Content-Length'] = (string) $length;
            }
        }
        // Set a default Content-Type if one was not supplied
        $type = $this->getHeader($headers, 'content-type');
        if (!$type && ($filename === '0' || $filename)) {
            if ($type = \PYS_PRO_GLOBAL\GuzzleHttp\Psr7\MimeType::fromFilename($filename)) {
                $headers['Content-Type'] = $type;
            }
        }
        return [$stream, $headers];
    }
    private function getHeader(array $headers, string $key)
    {
        $lowercaseHeader = \strtolower($key);
        foreach ($headers as $k => $v) {
            if (\strtolower($k) === $lowercaseHeader) {
                return $v;
            }
        }
        return null;
    }
}
